<?php

namespace App\Http\Controllers\Pay;

use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use App\Exceptions\RuleValidationException;
use App\Http\Controllers\PayController;
use Illuminate\Http\Request;

class HiiCashController extends PayController
{
    /**
     * HiiCash网关
     *
     * @param string $payway
     * @param string $orderSN
     */
    public function gateway(string $payway, string $orderSN)
    {
        try {
            // 加载网关
            $this->loadGateWay($orderSN, $payway);

            if (filled($this->order->trade_no)) {
                return $this->toPay($this->order->trade_no);
            }

            $order = [
                'mchOrderNo' => $this->order->order_sn,
                'amount'     => bcmul($this->order->total_price, $this->usdExchangeRate(), 2) * 100,
                'currency'   => 'USD',
                'notifyUrl'  => Str::startsWith(config('app.url'), 'https://') ? config('app.url') . '/pay/hiicash/notify' : 'https://' . config('app.url') . '/pay/hiicash/notify',
                'returnUrl'  => Str::startsWith(config('app.url'), 'https://') ? config('app.url') . '/detail-order-sn/' . $orderSN : 'https://' . config('app.url') . '/detail-order-sn/' . $orderSN,
                'subject'    => $this->order->title,
            ];

            $ret = $this->create($order);
            $this->order->trade_no = $ret['payOrderId'];
            $this->cachePayData($ret['payData']);
            $this->order->save();

            return $this->toPay($ret['payOrderId']);
        } catch (RuleValidationException $exception) {
            return $this->err($exception->getMessage());
        }
    }


    /**
     * 异步通知
     */
    public function notifyUrl(Request $request)
    {
        try {
            if (!$order = $this->orderService->detailOrderSN($request->post('mchOrderNo'))) {
                return;
            }
        } catch (\Exception $e) {
            return;
        }

        $amount = (float)($request->post('amount')) / 100;
        $trade_no = $request->post('channelOrderNo');
        $this->loadGateWay($order->order_sn, $order->pay->pay_check);

        $secretKey = $this->payGateway->merchant_pem;
        $body = $request->post();
        ksort($body);

        try {
            if ($this->check_notify($body, $request->header(), $secretKey)) {
                $this->orderProcessService->completedOrder($order->order_sn, $amount, $trade_no);
                return response()->json([
                    "returnCode" => "success",
                    "returnMsg"  => ""
                ]);
            }
        } catch (\Exception $e) {
            Log::log('debug', '验签失败' . $order->order_sn);
        }
    }

    // 美金汇率
    protected function usdExchangeRate(): float
    {
        return 1;
    }

    protected function create($order)
    {
        $mchNo = $this->payGateway->merchant_id;
        $appID = $this->payGateway->merchant_key;
        $secretKey = $this->payGateway->merchant_pem;

        $request = [
            "mchNo"       => $mchNo,
            "appId"       => $appID,
            "mchOrderNo"  => $order["mchOrderNo"],
            "amount"      => $order["amount"],
            "payDataType" => "Cashier",
            "currency"    => "USD",
            "subject"     => $order["subject"],
            "notifyUrl"   => $order["notifyUrl"],
            "returnUrl"   => $order["returnUrl"],
        ];

        ksort($request);

        $headers = [
            "HiicashPay-Timestamp" => (int)(string)floor(microtime(true) * 1000),
            "HiicashPay-Nonce"     => Str::random(32),
            "HiicashPay-AppId"     => $appID
        ];

        $body = json_encode($request, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $payload = $headers['HiicashPay-Timestamp'] . chr(0x0A) . $headers['HiicashPay-Nonce'] . chr(0x0A) . $body . chr(0x0A);
        $signature = $this->generate_signature($payload, $secretKey);
        $headers["HiicashPay-Signature"] = $signature;
        $jsonStr = json_encode($request, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $httpHeaders = [];
        foreach ($headers as $key => $header) {
            $httpHeaders[] = $key . ': ' . $header;
        }
        $httpHeaders[] = 'Content-Type: application/json; charset=utf-8';
        $httpHeaders[] = 'Content-Length: ' . strlen($jsonStr);
        $ch = curl_init(file_get_contents('https://hiicash.oss-ap-northeast-1.aliyuncs.com/gateway.txt') . 'pay/order/create');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonStr);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $httpHeaders);
        $response = curl_exec($ch);
        curl_close($ch);
        if (!$response) {
            abort(400, '网关无响应，请稍后重试');
        }
        $res = json_decode($response, true);
        if (!is_array($res) || !isset($res['data'])) {
            Log::error('HiiCash Payment Gateway Error：', $res);
            abort(400, '支付失败，请稍后再试');
        }
        if (!is_array($res['data']) || !isset($res['data']['payData'])) {
            Log::error('HiiCash Payment Gateway Error：', $res);
            abort(400, '支付失败，请稍后再试');
        }
        return $res['data'];
    }

    protected function check_notify(array $data, array $headers, $secretKey): bool
    {
        foreach ($data as $key => $datum) {
            $data[$key] = $datum ?? "";
        }
        return ($this->sign($data, $headers, $secretKey) === $this->header($headers, 'HiicashPay-Signature'));
    }

    protected function generate_signature(string $payload, $secretKey): string
    {
        // 使用 HMAC-SHA512 算法生成签名
        $hash = hash_hmac("sha512", $payload, $secretKey, true);

        // 将签名转换为大写字符串
        return strtoupper(bin2hex($hash));
    }

    protected function sign(array $data, array $headers, $secretKey): string
    {
        $body = json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $payload = $this->header($headers, 'HiicashPay-Timestamp') . "\n" .
            $this->header($headers, 'HiicashPay-Nonce') . "\n" . $body . "\n";

        return $this->generate_signature($payload, $secretKey);
    }

    protected function header($headers, $key)
    {
        $header = $headers[$key] ?? $headers[strtolower($key)];
        return is_array($header) ? $header[0] : $header;
    }

    protected function cachePayData($payData)
    {
        Cache::put($this->order->trade_no, $payData, 60 * 60);
    }

    protected function toPay($trade_no)
    {
        return redirect(Cache::get($trade_no));
    }
}
